import os
from typing import Dict, Type

import setuptools


MODULE_NAME = "excelrd"
REPOSITORY_URL = f"https://github.com/thombashi/{MODULE_NAME:s}"
REQUIREMENT_DIR = "requirements"

pkg_info: Dict[str, str] = {}


def get_release_command_class() -> Dict[str, Type[setuptools.Command]]:
    try:
        from releasecmd import ReleaseCommand
    except ImportError:
        return {}

    return {"release": ReleaseCommand}


with open(os.path.join(MODULE_NAME, "__version__.py")) as f:
    exec(f.read(), pkg_info)

with open("README.rst") as fp:
    long_description = fp.read()

with open(os.path.join(REQUIREMENT_DIR, "test_requirements.txt")) as f:
    TESTS_REQUIRES = [line.strip() for line in f if line.strip()]

with open(os.path.join(REQUIREMENT_DIR, "docs_requirements.txt")) as f:
    docs_requires = [line.strip() for line in f if line.strip()]

setuptools.setup(
    name=MODULE_NAME,
    version=pkg_info["__version__"],
    author=pkg_info["__author__"],
    author_email=pkg_info["__email__"],
    maintainer=pkg_info["__maintainer__"],
    maintainer_email=pkg_info["__maintainer_email__"],
    url=REPOSITORY_URL,
    packages=setuptools.find_packages(exclude=["test*"]),
    scripts=[
        "scripts/runxlrd.py",
    ],
    description=(
        "Library for developers to extract data from Microsoft Excel (tm) spreadsheet files"
    ),
    long_description=long_description,
    long_description_content_type="text/x-rst",
    license=pkg_info["__license__"],
    keywords=["xls", "excel", "spreadsheet", "workbook"],
    python_requires=">=3.7",
    tests_require=TESTS_REQUIRES,
    extras_require={
        "docs": docs_requires,
        "dev": ["releasecmd>=0.2.0,<1"] + TESTS_REQUIRES,
        "test": TESTS_REQUIRES,
    },
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: BSD License",
        "Programming Language :: Python",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3.7",
        "Programming Language :: Python :: 3.8",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Programming Language :: Python :: 3 :: Only",
        "Programming Language :: Python :: Implementation :: CPython",
        "Programming Language :: Python :: Implementation :: PyPy",
        "Operating System :: OS Independent",
        "Topic :: Office/Business",
        "Topic :: Office/Business :: Financial :: Spreadsheet",
        "Topic :: Software Development :: Libraries :: Python Modules",
    ],
    cmdclass=get_release_command_class(),
)
