/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { coreEvents } from '@google/gemini-cli-core';
// Capture the original stdout and stderr write methods before any monkey patching occurs.
const originalStdoutWrite = process.stdout.write.bind(process.stdout);
const originalStderrWrite = process.stderr.write.bind(process.stderr);
/**
 * Writes to the real stdout, bypassing any monkey patching on process.stdout.write.
 */
export function writeToStdout(...args) {
    return originalStdoutWrite(...args);
}
/**
 * Writes to the real stderr, bypassing any monkey patching on process.stderr.write.
 */
export function writeToStderr(...args) {
    return originalStderrWrite(...args);
}
/**
 * Monkey patches process.stdout.write and process.stderr.write to redirect output to the provided logger.
 * This prevents stray output from libraries (or the app itself) from corrupting the UI.
 * Returns a cleanup function that restores the original write methods.
 */
export function patchStdio() {
    const previousStdoutWrite = process.stdout.write;
    const previousStderrWrite = process.stderr.write;
    process.stdout.write = (chunk, encodingOrCb, cb) => {
        const encoding = typeof encodingOrCb === 'string' ? encodingOrCb : undefined;
        coreEvents.emitOutput(false, chunk, encoding);
        const callback = typeof encodingOrCb === 'function' ? encodingOrCb : cb;
        if (callback) {
            callback();
        }
        return true;
    };
    process.stderr.write = (chunk, encodingOrCb, cb) => {
        const encoding = typeof encodingOrCb === 'string' ? encodingOrCb : undefined;
        coreEvents.emitOutput(true, chunk, encoding);
        const callback = typeof encodingOrCb === 'function' ? encodingOrCb : cb;
        if (callback) {
            callback();
        }
        return true;
    };
    return () => {
        process.stdout.write = previousStdoutWrite;
        process.stderr.write = previousStderrWrite;
    };
}
/**
 * Creates proxies for process.stdout and process.stderr that use the real write methods
 * (writeToStdout and writeToStderr) bypassing any monkey patching.
 * This is used by Ink to render to the real output.
 */
export function createInkStdio() {
    const inkStdout = new Proxy(process.stdout, {
        get(target, prop, receiver) {
            if (prop === 'write') {
                return writeToStdout;
            }
            const value = Reflect.get(target, prop, receiver);
            if (typeof value === 'function') {
                return value.bind(target);
            }
            return value;
        },
    });
    const inkStderr = new Proxy(process.stderr, {
        get(target, prop, receiver) {
            if (prop === 'write') {
                return writeToStderr;
            }
            const value = Reflect.get(target, prop, receiver);
            if (typeof value === 'function') {
                return value.bind(target);
            }
            return value;
        },
    });
    return { stdout: inkStdout, stderr: inkStderr };
}
//# sourceMappingURL=stdio.js.map